//////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Software timer module (16-bit) - Header
//
// This module uses 16-bit counters to work better with 16-bit architectures.
// This means the softtimers can only count to 65536.  At 1ms, this would be
// 65,536 sec max
//
// Language: C
//
// File:     swtimer.h
// Author: S1axter - MyBitBox.com
// Created: 6/29/2013
//
// This file was created to be processor agnostic, no architecture specific code
// or hardware access should be included.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef __SWTIMER_H__
#define __SWTIMER_H__

#include "assert.h"

#define TIMERS_MAX  10

/*
 * swtimer_callback_t softtimer_A_callback();
 * swtimer_obj_t  soft_timer_A;
 *
 * // Start out at 1000 Hz
 * timer_setup(1, 7, TIMER_PRESCALE_1_64_, (oscillator_frequency_get()/64), 1000);
 * // Inside timer1 interrupt: swtimer_increment(1); // Kick the swtimer module every 1ms
 *
 * // Call function softtimer_A_callback every 500ms
 * swtimer_new(soft_timer_A, 500, softtimer_A_callback);
 * // Turn on the software timer
 * swtimer_on(soft_timer_A);
 *
 * // Call this in the main loop. Doesn't do anything until a tick is received
 * swtimer_process();
 *
 * // Callback
 * swtimer_callback_t softtimer_A_callback(){ // do things; return; }
 * // or
 * void softtimer_A_callback(){ // do things; return; }
 *
 *
*/



// Timeout callback prototype
typedef void(*swtimer_callback_t)();

typedef struct
{
    uint16_t period;
    uint16_t timecount;
    swtimer_callback_t callback;
    bool enabled;
}swtimer_obj_t;

typedef struct{
    uint16_t increment_value;
    uint8_t timers_max;
    uint8_t timers_index;
    swtimer_obj_t * list[TIMERS_MAX];
}swtimer_data_t;
extern swtimer_data_t g_swtimer_data;

#define swtimer_init()              do{g_swtimer_data.timers_max = TIMERS_MAX;}while(0)
#define swtimer_increment(count)    do{g_swtimer_data.increment_value+=count;}while(0);

#define swtimer_enabled(timer)              ((timer).enabled)
#define swtimer_on(timer)                   do{(timer).enabled = true;}while(0)
#define swtimer_off(timer)                  do{(timer).enabled = false;}while(0)
#define swtimer_enable(timer, enable)       do{(timer).enabled = (enable==true);}while(0)
#define swtimer_period_set(timer, per)      do{(timer).period = per;}while(0)

#define swtimer_callback_set(timer, callback_ptr)   do{(timer).callback = (swtimer_callback_t)callback_ptr;}while(0)

// timer is "swtimer_obj_t * timer"
#define swtimer_new_basic(timer, period, callback_ptr) \
do{                                              \
    swtimer_period_set(timer, period);           \
    swtimer_off(timer);                          \
    swtimer_callback_set(timer, callback_ptr);   \
}while(0)

// this is like swtimer_add(swtimer_obj_t * timer)
#define swtimer_add(timer)      \
do{                             \
    if(g_swtimer_data.timers_index>=(TIMERS_MAX-1)) assert(g_swtimer_data.timers_index);  \
    g_swtimer_data.list[g_swtimer_data.timers_index] = &timer;   \
    g_swtimer_data.timers_index++;                               \
}while(0);

#define swtimer_new(timer, period, callback_ptr) \
do{                                              \
    swtimer_period_set(timer, period);           \
    swtimer_off(timer);                          \
    swtimer_callback_set(timer, callback_ptr);   \
    swtimer_add(timer);                          \
}while(0)


// this is like swtimer_remove(swtimer_obj_t * timer)
// TODO: Make this
// TODO: Go to that id, shift everything above it down by 1, decrement size
/*
#define swtimer_remove(timer)      \
do{ \
    //if(g_swtimer_data.timers_index<(TIMERS_MAX-1)) assert(g_swtimer_data.timers_index);  \
    //g_swtimer_data.list[g_swtimer_data.timers_index] = timer;   \
    //g_swtimer_data.timers_index++;                              \
    }       \
}while(0);
*/

void swtimer_process();

#endif
